<?php
error_reporting(E_ALL);
ini_set('display_errors', 0);
include 'db_connect.php';
header("Content-Type: application/json; charset=UTF-8");

$user_id = $_GET['user_id'] ?? null;
if (!$user_id) {
    echo json_encode(["status" => "error", "message" => "User ID is required"]);
    exit;
}

/* -------------------- Helpers -------------------- */
function map_by($rows, $key) {
    $out = [];
    foreach ($rows as $r) {
        $out[$r[$key]] = $r;
    }
    return $out;
}
function group_by($rows, $key) {
    $out = [];
    foreach ($rows as $r) {
        $out[$r[$key]][] = $r;
    }
    return $out;
}
function ints_csv($arr) {
    return implode(',', array_map('intval', $arr));
}

/* -------------------- Step 0: root user referCode -------------------- */
$referCode = null;
$stmt = $conn->prepare("SELECT referCode FROM sign_up WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($referCode);
$stmt->fetch();
$stmt->close();

if (!$referCode) {
    echo json_encode(["status" => "error", "message" => "User not found"]);
    exit;
}

/* -------------------- Step 1: Level 1 users -------------------- */
$sql = "SELECT id, name, referCode, referredBy FROM sign_up WHERE referredBy = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("s", $referCode);
$stmt->execute();
$res = $stmt->get_result();
$level1 = $res->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$level1_ids = array_column($level1, 'id');
$level1_codes = array_column($level1, 'referCode');

/* যদি L1 ফাঁকা হয়, দ্রুত রেসপন্স */
if (empty($level1)) {
    // নিজের orders লাগবে
    $orderCount = 0;
    $stmt = $conn->prepare("SELECT COUNT(*) FROM orders WHERE user_id = ? AND order_status IN ('Delivered') 
                    AND created_at > (SELECT COALESCE(MAX(claimed_at), '1970-01-01') 
                    FROM leadership_claims 
                    WHERE user_id = ? AND leadership_type = 'Silver')");
    $stmt->bind_param("ii", $user_id, $user_id);
    $stmt->execute();
    $stmt->bind_result($orderCount);
    $stmt->fetch();
    $stmt->close();

    echo json_encode([
        "status" => "success",
        "user_id" => $user_id,
        "reward" => "None",
        "orders" => $orderCount,
        "level1_summary" => [
            "total_referred" => 0,
            "verified" => 0,
            "leaders" => 0,
            "remaining_to_leader" => 15
        ],
        "silver_summary" => ["total_candidates"=>0,"times"=>0,"total_orders"=>$orderCount,"reward"=>null],
        "gold_summary"   => ["total_candidates"=>0,"times"=>0,"total_orders"=>$orderCount,"reward"=>null],
        "diamond_summary"=> ["total_candidates"=>0,"times"=>0,"total_orders"=>$orderCount,"reward"=>null],
        "top_summary"    => ["total_candidates"=>0,"times"=>0,"total_orders"=>$orderCount,"reward"=>null],
        "silver_candidates"=>[],
        "gold_candidates"=>[],
        "diamond_candidates"=>[],
        "top_candidates"=>[],
        "reward_breakdown"=>[
            "Rootva Leader"=>0,"Silver"=>0,"Gold"=>0,"Diamond"=>0,"Top"=>0
        ],
        "user_level_rewards"=>[
            "Rootva Leader"=>0,"Silver"=>0,"Gold"=>0,"Diamond"=>0,"Top"=>0
        ]
    ], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
    exit;
}

/* -------------------- Step 2: Level 2 users (referredBy in L1 referCodes) -------------------- */
$in_codes = "'" . implode("','", array_map([$conn, 'real_escape_string'], $level1_codes)) . "'";
$l2_sql = "SELECT id, name, referCode, referredBy FROM sign_up WHERE referredBy IN ($in_codes)";
$l2_res = $conn->query($l2_sql);
$level2 = $l2_res ? $l2_res->fetch_all(MYSQLI_ASSOC) : [];
$l2_by_parent_code = group_by($level2, 'referredBy');

$level2_ids = array_column($level2, 'id');
$level2_codes = array_column($level2, 'referCode');

/* -------------------- Step 3: Level 3 users (referredBy in L2 referCodes) -------------------- */
if (!empty($level2_codes)) {
    $in_codes2 = "'" . implode("','", array_map([$conn, 'real_escape_string'], $level2_codes)) . "'";
    $l3_sql = "SELECT id, name, referCode, referredBy FROM sign_up WHERE referredBy IN ($in_codes2)";
    $l3_res = $conn->query($l3_sql);
    $level3 = $l3_res ? $l3_res->fetch_all(MYSQLI_ASSOC) : [];
} else {
    $level3 = [];
}
$l3_by_parent_code = group_by($level3, 'referredBy');
$level3_ids = array_column($level3, 'id');

/* -------------------- Step 4: Bulk counts (verification + orders) -------------------- */
$all_ids = array_unique(array_merge([$user_id], $level1_ids, $level2_ids, $level3_ids));
$csv_ids = empty($all_ids) ? '0' : ints_csv($all_ids);

/* verification counts */
$verif_sql = "
    SELECT user_id, COUNT(*) AS c
    FROM verification_requests
    WHERE status='Approved' AND user_id IN ($csv_ids)
    GROUP BY user_id
";
$verif_res = $conn->query($verif_sql);
$verified_map = [];
if ($verif_res) {
    while ($r = $verif_res->fetch_assoc()) {
        $verified_map[(int)$r['user_id']] = (int)$r['c'];
    }
}

/* orders counts (Delivered) */
$orders_sql = "
    SELECT user_id, COUNT(*) AS c
    FROM orders
    WHERE order_status = 'Delivered' 
      AND created_at > (
          SELECT COALESCE(MAX(claimed_at), '1970-01-01') 
          FROM leadership_claims 
          WHERE user_id = ? AND leadership_type = 'Silver'
      )
      AND user_id IN ($csv_ids)
    GROUP BY user_id
";

$stmt = $conn->prepare($orders_sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$orders_res = $stmt->get_result();

$orders_map = [];
if ($orders_res) {
    while ($r = $orders_res->fetch_assoc()) {
        $orders_map[(int)$r['user_id']] = (int)$r['c'];
    }
}
$orderCount = $orders_map[$user_id] ?? 0;

/* -------------------- Step 5: Compute L1 verified/self/leader -------------------- */
$verifiedCount = 0;   // L1 verified self
$leaderCount   = 0;   // L1 যারা leader (>=15 verified L2)
$level1Details = [];
$silverCandidates = [];
$goldCandidates   = [];
$diamondCandidates= [];
$topCandidates    = [];

foreach ($level1 as $m) {
    $member_id   = (int)$m['id'];
    $member_name = $m['name'];
    $member_code = $m['referCode'];

    /* L2 under this L1 */
    $l2_under = $l2_by_parent_code[$member_code] ?? [];

    /* verified L2 = L2 child যাদের নিজে Approved>0 */
    $verifiedL2 = 0;
    foreach ($l2_under as $l2) {
        $lid = (int)$l2['id'];
        if (($verified_map[$lid] ?? 0) > 0) $verifiedL2++;
    }

    $leaderMultiplier = intdiv($verifiedL2, 15);
    $isLeader = $leaderMultiplier >= 1;
    if ($isLeader) {
        $leaderCount += 1; // একবারই গণনা
        $silverCandidates[] = [
            "id" => $member_id,
            "name" => $member_name,
            "verified_referrals" => $verifiedL2,
            "is_leader" => true,
            "leader_times" => $leaderMultiplier
        ];
    }

    /* L1 member নিজে verified কিনা */
    if (($verified_map[$member_id] ?? 0) > 0) $verifiedCount++;

    $level1Details[] = [
        "id" => $member_id,
        "name" => $member_name,
        "verified_referrals" => $verifiedL2,
        "is_leader" => $isLeader,
        "leader_times" => $leaderMultiplier
    ];
}

$remainingToLeader = max(0, 15 - $verifiedCount);

/* -------------------- Step 6: Evaluate Silver/Gold/Diamond/Top candidates -------------------- */
/*
    SilverCandidates = যেসব L1 leader
    subLeaders = ওই candidate-এর L2 সাবদের (root এর L2) মধ্যে যারা তাদের নিজস্ব L2 (root এর L3) verified>=15
*/
$silverOrdersTotal = 0;
$goldOrdersTotal   = 0;
$diamondOrdersTotal= 0;
$topOrdersTotal    = 0;

foreach ($silverCandidates as $cand) {
    $cand_id   = (int)$cand['id'];
    $cand_code = null;
    // code আনার বদলে, level1 array থেকেই বের করি
    // (loop এ map_by লাগাতে চাইলে আলাদা map বানাতে পারো)
    foreach ($level1 as $m) {
        if ((int)$m['id'] === $cand_id) { $cand_code = $m['referCode']; break; }
    }
    if (!$cand_code) continue;

    $candOrderCount = $orders_map[$cand_id] ?? 0;
    $silverOrdersTotal += $candOrderCount;

    // Candidate-এর সাবরা = root এর L2 যাদের referredBy = cand_code
    $subs = $l2_by_parent_code[$cand_code] ?? [];

    $subLeaders = 0;
    foreach ($subs as $sub) {
        $sub_id   = (int)$sub['id'];
        $sub_code = $sub['referCode'];

        // sub-এর L2 (root এর L3) যারা referredBy = sub_code
        $lvl2s_of_sub = $l3_by_parent_code[$sub_code] ?? [];

        // ওই lvl2s_of_sub এর মধ্যে কতজন verified>0
        $subLeaderVerified = 0;
        foreach ($lvl2s_of_sub as $u3) {
            $u3_id = (int)$u3['id'];
            if (($verified_map[$u3_id] ?? 0) > 0) $subLeaderVerified++;
        }
        if ($subLeaderVerified >= 15) $subLeaders++;
    }

    // ক্যাটেগরি rule
    if ($subLeaders >= 10) {
        $goldCandidates[] = $cand;
        $goldOrdersTotal += $candOrderCount;
    }
    if ($subLeaders >= 10 && $orderCount >= 3) {
        $diamondCandidates[] = $cand;
        $diamondOrdersTotal += $candOrderCount;
    }
    if ($subLeaders >= 10 && $orderCount >= 10) {
        $topCandidates[] = $cand;
        $topOrdersTotal += $candOrderCount;
    }
}

/* -------------------- Step 7: Reward times -------------------- */
$leaderTimes  = intdiv($verifiedCount, 15);
$silverTimes  = ($orderCount >= 1)  ? intdiv($leaderCount, 10) : 0;
$goldTimes    = (count($goldCandidates)    >= 10 && $orderCount >= 3)  ? intdiv(count($goldCandidates), 10) : 0;
$diamondTimes = (count($diamondCandidates) >= 10 && $orderCount >= 5)  ? intdiv(count($diamondCandidates), 10) : 0;
$topTimes     = (count($topCandidates)     >= 10 && $orderCount >= 10) ? intdiv(count($topCandidates), 10) : 0;

$reward = "None";
if ($topTimes     >= 1) $reward = "Top ".$topTimes;
else if ($diamondTimes>=1) $reward = "Diamond ".$diamondTimes;
else if ($goldTimes  >= 1) $reward = "Gold ".$goldTimes;
else if ($silverTimes>= 1) $reward = "Silver ".$silverTimes;
else if ($leaderTimes>= 1) $reward = "Rootva Leader ".$leaderTimes;

$amounts = [
    "Rootva Leader" => 80,
    "Silver"  => 500,
    "Gold"    => 1000,
    "Diamond" => 2000,
    "Top"     => 4000
];

$current_time = date('d-m-Y h:i A');

/* -------------------- Step 8: Auto-insert Leader transactions (new times only) -------------------- */
if ($leaderTimes > 0) {
    $stmt = $conn->prepare("
        SELECT COUNT(*) 
        FROM transactions 
        WHERE user_id = ? AND description = 'Rootva Leader reward'
    ");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $stmt->bind_result($alreadyGivenTimes);
    $stmt->fetch();
    $stmt->close();

    if ($leaderTimes > $alreadyGivenTimes) {
        $newTimes = $leaderTimes - $alreadyGivenTimes;

        $stmt = $conn->prepare("
            INSERT INTO transactions 
            (user_id, refer_id, amount, type, payment_gateway, description, update_at, created_at, date) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");

        $refer_id = $referCode; 
        $amount = $amounts["Rootva Leader"];
        $type = "income";
        $payment_gateway = "Rootva Leader Reward";
        $description = "Rootva Leader reward";

        for ($i = 0; $i < $newTimes; $i++) {
            $stmt->bind_param("isdsssss", $user_id, $refer_id, $amount, $type, $payment_gateway, $description, $current_time, $current_time);
            $stmt->execute();
        }
        $stmt->close();
    }
}

/* -------------------- Step 9: Create/ensure requests for Silver/Gold/Diamond/Top -------------------- */
$leaderships = [
    "Silver"  => $silverTimes,
    "Gold"    => $goldTimes,
    "Diamond" => $diamondTimes,
    "Top"     => $topTimes
];

foreach ($leaderships as $type => $times) {
    if ($times <= 0) continue;

    // Pending আছে কি না
    $stmt = $conn->prepare("
        SELECT COUNT(*) 
        FROM leadership_reward_requests 
        WHERE user_id = ? AND reward_type = ? AND status = 'Pending'
    ");
    $stmt->bind_param("is", $user_id, $type);
    $stmt->execute();
    $stmt->bind_result($pendingExists);
    $stmt->fetch();
    $stmt->close();
    if ($pendingExists > 0) continue;

    // Approved times sum
    $stmt = $conn->prepare("
        SELECT COALESCE(SUM(times),0) 
        FROM leadership_reward_requests 
        WHERE user_id = ? AND reward_type = ? AND status = 'Approved'
    ");
    $stmt->bind_param("is", $user_id, $type);
    $stmt->execute();
    $stmt->bind_result($totalApprovedTimes);
    $stmt->fetch();
    $stmt->close();

    if ($times > $totalApprovedTimes) {
        $newTimes = $times - $totalApprovedTimes;
        $amount = $amounts[$type] * $newTimes;

        $stmt = $conn->prepare("
            INSERT INTO leadership_reward_requests 
            (user_id, reward_type, times, amount, status) 
            VALUES (?, ?, ?, ?, 'Pending')
        ");
        $stmt->bind_param("isid", $user_id, $type, $newTimes, $amount);
        $stmt->execute();
        $stmt->close();
    }
}

/* -------------------- Step 10: Labels & personal rewards -------------------- */
function personalRewardLabel($leader_times) {
    if ($leader_times <= 0) return null;
    return "Rootva Leader ×".intval($leader_times);
}

foreach ($level1Details as &$d) {
    $d['personal_rewards'] = [];
    if (!empty($d['leader_times']) && $d['leader_times'] > 0) {
        $d['personal_rewards'][] = personalRewardLabel($d['leader_times']);
    }
}
unset($d);

foreach (['silverCandidates','goldCandidates','diamondCandidates','topCandidates'] as $listName) {
    foreach ($$listName as $i => $cand) {
        $$listName[$i]['personal_rewards'] = [];
        if (!empty($cand['leader_times']) && $cand['leader_times'] > 0) {
            $$listName[$i]['personal_rewards'][] = personalRewardLabel($cand['leader_times']);
        }
    }
}

/* -------------------- Step 11: Summaries -------------------- */
$user_level_rewards = [
    "Rootva Leader" => $leaderTimes,
    "Silver"  => $silverTimes,
    "Gold"    => $goldTimes,
    "Diamond" => $diamondTimes,
    "Top"     => $topTimes
];

$level1_reward_label  = ($leaderTimes  >= 1) ? "Rootva Leader ".$leaderTimes : null;
$silver_reward_label  = ($silverTimes  >= 1) ? "Silver ".$silverTimes       : null;
$gold_reward_label    = ($goldTimes    >= 1) ? "Gold ".$goldTimes           : null;
$diamond_reward_label = ($diamondTimes >= 1) ? "Diamond ".$diamondTimes     : null;
$top_reward_label     = ($topTimes     >= 1) ? "Top ".$topTimes             : null;

/* -------------------- Final Output -------------------- */
echo json_encode([
    "status" => "success",
    "user_id" => $user_id,
    "reward" => $reward,
    "orders" => $orderCount,

    "level1_summary" => [
        "total_referred" => count($level1),
        "verified" => $verifiedCount,
        "leaders" => $leaderCount,
        "remaining_to_leader" => $remainingToLeader,
        "reward" => $level1_reward_label
    ],

    "silver_summary" => [
        "total_candidates" => count($silverCandidates),
        "times" => $silverTimes,
        "total_orders" => $orderCount,
        "reward" => $silver_reward_label
    ],
    "gold_summary" => [
        "total_candidates" => count($goldCandidates),
        "times" => $goldTimes,
        "total_orders" => $orderCount,
        "reward" => $gold_reward_label
    ],
    "diamond_summary" => [
        "total_candidates" => count($diamondCandidates),
        "times" => $diamondTimes,
        "total_orders" => $orderCount,
        "reward" => $diamond_reward_label
    ],
    "top_summary" => [
        "total_candidates" => count($topCandidates),
        "times" => $topTimes,
        "total_orders" => $orderCount,
        "reward" => $top_reward_label
    ],

    //"level1_details" => $level1Details, // চাইলে পাঠাও
    "silver_candidates" => $silverCandidates,
    "gold_candidates"   => $goldCandidates,
    "diamond_candidates"=> $diamondCandidates,
    "top_candidates"    => $topCandidates,

    "reward_breakdown" => [
        "Rootva Leader" => $leaderTimes,
        "Silver" => $silverTimes,
        "Gold" => $goldTimes,
        "Diamond" => $diamondTimes,
        "Top" => $topTimes
    ],
    "user_level_rewards" => $user_level_rewards
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
?>