<?php
// ------------------------------------------------------------
// Fast Referral Tree API (up to 7 levels) — Optimized Version
// ------------------------------------------------------------
declare(strict_types=1);

ob_start();
include 'db_connect.php';
$conn->set_charset('utf8mb4');
ob_clean();
header('Content-Type: application/json; charset=utf-8');

$start_time = microtime(true);

// -------- Input --------
$referCode  = $_GET['referCode'] ?? null;
$isUpdated  = isset($_GET['isUpdated']) && $_GET['isUpdated'] === 'true';
$limit      = $isUpdated ? max(0, (int)($_GET['limit']  ?? 20)) : 0;
$offset     = $isUpdated ? max(0, (int)($_GET['offset'] ?? 0))  : 0;

$response = [];

if (!$referCode) {
    echo json_encode(["message" => "অবৈধ ডেটা"], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Run a single level batched query:
 * SELECT ... FROM sign_up WHERE referredBy IN (?,?,...)
 * returns array of rows with an extra 'level'
 */
function fetch_level_users_by_referredBy_codes(mysqli $conn, array $parentCodes, int $level): array {
    if (empty($parentCodes)) return [];

    // dynamic placeholders
    $placeholders = implode(',', array_fill(0, count($parentCodes), '?'));
    $types = str_repeat('s', count($parentCodes));

    $sql = "
        SELECT id, name, number, referCode, is_verified, profile_pic_url, verified_at, created_at
        FROM sign_up
        WHERE referredBy IN ($placeholders)
    ";
    $stmt = $conn->prepare($sql);
    if (!$stmt) throw new RuntimeException("Prepare failed: ".$conn->error);

    $stmt->bind_param($types, ...$parentCodes);
    $stmt->execute();
    $res = $stmt->get_result();

    $rows = [];
    while ($row = $res->fetch_assoc()) {
        $row['level'] = $level;
        $rows[] = $row;
    }
    $stmt->close();
    return $rows;
}

/**
 * BFS (level-wise) up to $maxLevel.
 * Returns: flat array of all users (each has 'level')
 * Also returns: levels grouped if requested.
 */
function bfs_referrals_all_levels(mysqli $conn, string $rootReferCode, int $maxLevel = 7): array {
    $allUsers = [];
    $currentLevel = 1;
    $queue = [$rootReferCode];

    while (!empty($queue) && $currentLevel <= $maxLevel) {
        // ১টা কুয়েরিতে এই লেভেলের সব চাইল্ড
        $levelRows = fetch_level_users_by_referredBy_codes($conn, $queue, $currentLevel);
        if (empty($levelRows)) break;

        $allUsers = array_merge($allUsers, $levelRows);

        // পরের লেভেলের parent codes = এই লেভেলের referCode গুলো
        $queue = array_column($levelRows, 'referCode');
        $currentLevel++;
    }
    return $allUsers;
}

try {
    // একবারেই ৭ লেভেল পর্যন্ত এনে নিলাম (সর্বোচ্চ ৭টা কুয়েরি)
    $allUsers = bfs_referrals_all_levels($conn, $referCode, 7);

    // গ্লোবাল DESC (id) বজায় রাখতে এখানে sort
    usort($allUsers, fn($a, $b) => ($b['id'] <=> $a['id']));

    if ($isUpdated) {
        // Flat + pagination
        $total = count($allUsers);
        $page  = array_slice($allUsers, $offset, $limit);

        $response['data']    = [['users' => $page]];
        $response['total']   = $total;
        $response['hasMore'] = ($offset + $limit) < $total;
    } else {
        // Level grouped output (পুরনো ফরম্যাট, কিন্তু ফাস্ট BFS থেকে গ্রুপিং)
        $levels = [];
        if (!empty($allUsers)) {
            // group by level, তারপর প্রতিটা লেভেলেও (id DESC) – optional
            $byLevel = [];
            foreach ($allUsers as $row) {
                $lvl = (int)$row['level'];
                $byLevel[$lvl][] = $row;
            }
            ksort($byLevel); // 1..7

            foreach ($byLevel as $lvl => $rows) {
                usort($rows, fn($a, $b) => ($b['id'] <=> $a['id']));
                $levels[] = [
                    'level' => $lvl,
                    'users' => $rows
                ];
            }
        }
        $response['data'] = $levels;
    }

    $response['message'] = "ডেটা সফলভাবে লোড হয়েছে";
} catch (Throwable $e) {
    $response = [
        "message" => "সার্ভার ত্রুটি: " . $e->getMessage()
    ];
}

// Timing
$end_time = microtime(true);
$response['load_time'] = round($end_time - $start_time, 4) . " sec";

// Safe JSON output
$json = json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
if ($json === false) {
    $json = json_encode([
        "message" => "JSON Encode ত্রুটি: " . json_last_error_msg(),
        "raw_data_length" => strlen(json_encode($response))
    ]);
}

echo $json;
?>