<?php
header('Content-Type: application/json; charset=utf-8');
include 'db_connect.php';

$conn->set_charset("utf8mb4");
date_default_timezone_set("Asia/Dhaka");

// Input validation
$user_id = intval($_POST['user_id'] ?? 0);

if ($user_id <= 0) {
    echo json_encode(['error' => true, 'message' => 'Invalid user ID']);
    exit;
}

try {
    // Start transaction for data consistency
    $conn->begin_transaction();

    // Check if user exists in wheel_spin_info
    $stmt = $conn->prepare("SELECT * FROM wheel_spin_info WHERE user_id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        // Create new record for new user
        $current_time = date('Y-m-d H:i:s');
        
        // Fixed column name (last_spin_at instead of last_spin_at)
        $stmt = $conn->prepare("INSERT INTO wheel_spin_info 
                              (user_id, total_spin, free_spin_used, spin_balance, last_spin_at, claimed) 
                              VALUES (?, 0, 0, 0.00, ?, 0)");
        $stmt->bind_param("is", $user_id, $current_time);
        $stmt->execute();
        
        $response = [
            'error' => false,
            'message' => 'New user spin info created',
            'spin_balance' => 0,
            'total_spin' => 0,
            'free_spin_used' => 0,
            'last_spin_at' => $current_time,
            'claimed' => false
        ];
    } else {
        // Return existing user data
        $info = $result->fetch_assoc();
        
        $response = [
            'error' => false,
            'message' => 'Spin info loaded',
            'spin_balance' => (float)$info['spin_balance'], // Ensure decimal type
            'total_spin' => (int)$info['total_spin'],
            'free_spin_used' => (int)$info['free_spin_used'],
            'last_spin_at' => $info['last_spin_at'],
            'claimed' => (bool)$info['claimed'] // Convert to boolean
        ];
    }

    $conn->commit();
    echo json_encode($response);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode([
        'error' => true,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} finally {
    // Ensure connection is properly closed
    if (isset($stmt)) {
        $stmt->close();
    }
}
?>