<?php
header('Content-Type: application/json; charset=utf-8');
include 'db_connect.php';

$conn->set_charset("utf8mb4");
date_default_timezone_set("Asia/Dhaka");

// Input validation
$user_id = intval($_POST['user_id'] ?? 0);
$amount = intval($_POST['amount'] ?? 0);

if ($user_id <= 0 || $amount <= 0) {
    echo json_encode(['error' => true, 'message' => 'Invalid input']);
    exit;
}

try {
    $conn->autocommit(FALSE); // Start transaction

    // Check user exists
    $stmt = $conn->prepare("SELECT id FROM sign_up WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    if ($stmt->get_result()->num_rows === 0) {
        throw new Exception("User not found");
    }

    // Get or insert wheel_spin_info
    $stmt = $conn->prepare("SELECT * FROM wheel_spin_info WHERE user_id = ? FOR UPDATE");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $info = $stmt->get_result()->fetch_assoc();
    
    if (!$info) {
        $stmt = $conn->prepare("INSERT INTO wheel_spin_info (user_id, total_spin, free_spin_used, spin_balance, last_spin_at, claimed) 
                              VALUES (?, 0, 0, 0.00, NULL, 0)");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();

        $stmt = $conn->prepare("SELECT * FROM wheel_spin_info WHERE user_id = ? FOR UPDATE");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $info = $stmt->get_result()->fetch_assoc();
    }

    // 🛑 Check if spin bonus already claimed
    if ($info['claimed'] == 1) {
        throw new Exception("আপনি ইতিমধ্যেই স্পিন বোনাস ক্লেইম করেছেন");
    }

    $canSpin = false;
    $free_spin = false;

    // Free spin check
    if ($info['free_spin_used'] < 5) {
        $canSpin = true;
        $free_spin = true;
    } else {
        // Check verification
        $stmt = $conn->prepare("SELECT id FROM verification_requests WHERE user_id = ? AND status = 'Approved'");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        if ($stmt->get_result()->num_rows === 0) {
            throw new Exception("Please verify your account to continue spinning");
        }

        // Cooldown check
        if (!empty($info['last_spin_at'])) {
            $lastSpin = strtotime($info['last_spin_at']);
            $remaining = (6 * 3600) - (time() - $lastSpin);
            if ($remaining > 0) {
                $minutes = ceil($remaining / 60);
                throw new Exception("Please wait $minutes minutes before spinning again");
            }
        }

        $canSpin = true;
    }

    if ($canSpin) {
        $new_balance = $info['spin_balance'] + $amount;
        $total_spins = $info['total_spin'] + 1;
        $free_spins_used = $free_spin ? ($info['free_spin_used'] + 1) : $info['free_spin_used'];
        $is_free_spin = $free_spin ? 1 : 0;
        $current_time = date('Y-m-d H:i:s');

        // Update wheel_spin_info (including claimed = 1)
        $stmt = $conn->prepare("UPDATE wheel_spin_info SET 
            total_spin = ?, 
            free_spin_used = ?, 
            spin_balance = ?, 
            last_spin_at = ?
            WHERE user_id = ?");
        $stmt->bind_param("iidsi", $total_spins, $free_spins_used, $new_balance, $current_time, $user_id);
        $stmt->execute();

        // Insert into spin_history
        $stmt = $conn->prepare("INSERT INTO spin_history (user_id, amount, is_free_spin, created_at) 
                              VALUES (?, ?, ?, ?)");
        $stmt->bind_param("iiis", $user_id, $amount, $is_free_spin, $current_time);
        $stmt->execute();

        $conn->commit();

        echo json_encode([
            'error' => false,
            'message' => 'Spin Bonus Added',
            'amount' => $amount,
            'total_balance' => $new_balance,
            'free_spins_used' => $free_spins_used,
            'is_free_spin' => $free_spin,
            'last_spin_at' => $current_time
        ]);
    }
} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(['error' => true, 'message' => $e->getMessage()]);
} finally {
    $conn->autocommit(TRUE);
}
?>
